<?php

namespace App\Http\Controllers\Backend\Covid;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\Backend\BookingRequest;
use Torann\GeoIP\Facades\GeoIP;

// use components
use Carbon\Carbon,
	Config,
	Session,
	App\Helper;

// use services
use App\Services\ServiceInterface,
	App\Services\MasterInterface,
	App\Services\BookingInterface;

class BookingController extends Controller
{
	protected
		$service,
		$master,
		$booking;

	public function __construct(
		ServiceInterface $service,
		MasterInterface $master,
		BookingInterface $booking
	) {
		parent::__construct();

		$this->service = $service;
		$this->master = $master;
		$this->booking = $booking;
	}

	/**
	 * Display a listing of the resource.
	 *
	 * @param int $yy
	 * @param int $mm
	 * @param int $dd
	 * @param int $is_pm
	 */
	public function index(int $yy = null, int $mm = null, int $dd = null, int $is_pm = null)
	{
		// initialized
		$data = [];

		// get parameeters
		$data = [
			'yy' => (is_null($yy)) ? date('Y') : $yy,
			'mm' => (is_null($mm)) ? date('n') : $mm,
			'dd' => (is_null($dd)) ? date('j') : $dd,
			'is_pm' => (is_null($is_pm)) ? 0 : $is_pm,
		];
		//echo '<pre>';print_r($data);die();

		// get date
		$data['date'] = $this->master->getDate(
			$keys = [
				['mtb_date.yy', '=', $data['yy']],
				['mtb_date.mm', '=', $data['mm']],
				['mtb_date.dd', '=', $data['dd']],
			]
		);
		//echo '<pre>';print_r($data['date']);die();

		if(is_null($data['date'])) $data['date'] = [];

		// set date
		$data['date'] = array_merge($data['date'], [
			'prev' => Helper::getDate(Carbon::create($data['yy'], $data['mm'], $data['dd'])->subDay(1)->toDateString()),
			'next' => Helper::getDate(Carbon::create($data['yy'], $data['mm'], $data['dd'])->addDay(1)->toDateString()),
		]);
		//echo '<pre>';print_r($data['date']);die();

		// get count
		$data['count'] = [
			'accepted_numbers' => $this->master->getField(
				$field = 'accepted_numbers',
				$keys = [
					['yy', '=', $data['yy']],
					['mm', '=', $data['mm']],
					['dd', '=', $data['dd']],
				],
				$table = 'mtb_date'
			),
			'booked_numbers' => $this->booking->getCount(
				$keys = [
					['yy', '=', $data['yy']],
					['mm', '=', $data['mm']],
					['dd', '=', $data['dd']],
					['is_booked', '=', 1],
				],
				$table = 'dtb_booking'
			),
		];
		//echo '<pre>';print_r($data['count']);die();

		// set remain_numbers
		$data['count']['remain_numbers'] = $data['count']['accepted_numbers'] - $data['count']['booked_numbers'];
		//echo '<pre>';print_r($data['count']);die();

		// get time_list
		$data['time_list'] = $this->master->getTimeDropdownList(
			$keys = [
				['yy', '=', $data['yy']],
				['mm', '=', $data['mm']],
				['dd', '=', $data['dd']],
			],
			$orders = [
				'dd' => 'ASC',
			]
		);
		//echo '<pre>';print_r($data['time_list']);die();

		// get booking_list
		$booking_list = $this->booking->getBookingList(
			$keys = [
				['yy', '=', $data['yy']],
				['mm', '=', $data['mm']],
				['dd', '=', $data['dd']],
				['is_booked', '=', 1],
			],
			$orders = [
				'dd' => 'ASC',
				'hh' => 'ASC',
				'ii' => 'ASC',
			],
			$options = []
		);
		//echo '<pre>';print_r($booking_list);die();

		// get booking_list
		$data['booking_list'] = $this->booking->setBookingList(
			$data['date'],
			$data['time_list'],
			$booking_list
		);
		//echo '<pre>';print_r($data['booking_list']);die();

		// get booking_list
		$data['booking_list_count'] = count($data['booking_list'][0]);
		//echo '<pre>';print_r($data['booking_list_count']);die();

		// set title
		$data['title'] = __("lang.booking.index");

		// get lang
		$data['lang'] = $this->lang;
		//echo '<pre>';print_r($data['lang']);die();

		// set tab
		$data['tab'] = 'booking';

		//echo '<pre>';print_r($data);die();
		return view('backend.covid.booking.index', $data);
	}

	/**
	 * Store a newly created resource in storage.
	 */
	public function store(BookingRequest $request)
	{
		// initialized
		$data = [];

		// get request
		$data = $request->all();
		//echo '<pre>';print_r($data);die();

		if($data['booking']['is_staff'] == 0 && $data['booking']['is_disease'] == 0) {
			//die('here');

			// get date_booking
			$date_booking = Carbon::create(
				$data['booking']['yy'],
				$data['booking']['mm'],
				$data['booking']['dd']
			);
			//echo '<pre>';print_r($date_booking);die();

			// get birthday
			$birthday = Carbon::create(
				$data['birthday']['yy'],
				$data['birthday']['mm'],
				$data['birthday']['dd']
			);
			//echo '<pre>';print_r($birthday);die();

			// get date_last
			$date_last = Carbon::create(
				$data['date_last']['yy'],
				$data['date_last']['mm'],
				$data['date_last']['dd']
			);
			//echo '<pre>';print_r($date_last);die();

			// get return as below
			if(Helper::getIsDisabledAge(
				$date_booking,
				$birthday
			)) {
				return redirect()->route('admin.covid.booking.update', [
						$data['booking']['yy'],
						$data['booking']['mm'],
						$data['booking']['dd'],
						$data['booking']['mtb_vaccine_id'],
					])
					->withInput()
					->withErrors(__("lang.error.disabled_age"));
			}

			// get return as below
			if(Helper::getIsAvailableDate(
				$date_booking,
				$date_last
			)) {
				return redirect()->route('admin.covid.booking.update', [
						$data['booking']['yy'],
						$data['booking']['mm'],
						$data['booking']['dd'],
						$data['booking']['mtb_vaccine_id'],
					])
					->withInput()
					->withErrors(__("lang.error.available_date"));
			}

			// get return as below
			//if(Helper::getIsSpecialCase(
			//	$date_booking,
			//	$birthday,
			//	$data['booking']
			//)) {
			//	return redirect()->route('admin.covid.booking.update', [
			//			$data['booking']['yy'],
			//			$data['booking']['mm'],
			//			$data['booking']['dd'],
			//			$data['booking']['mtb_vaccine_id'],
			//		])
			//		->withInput()
			//		->withErrors(__("lang.error.special_case"));
			//}
		}

		// get agent
		$data['agent'] = Helper::getAgent($request);
		//echo '<pre>';print_r($data['agent']);die();

		/**
		 * --------------------------------------------------
		 * transaction
		 * --------------------------------------------------
		 */
		$data['booking'] = \DB::transaction(function() use($data) {
			//echo '<pre>';print_r($data);die();

			// store processing
			$data['booking'] = $this->booking->storeBooking($data);
			//echo '<pre>';print_r($data['booking']);die();

			if(!is_null($data['booking']['mobile'])) {
				//die('mobile');

				// send sms
				// $sms = $this->booking->sendSms(
				// 	$data['booking']
				// );
			}

			//echo '<pre>';print_r($data['booking']);die();
			return $data['booking'];
		});

		// redirect
		return redirect()->route('admin.covid.booking.index', [
			$data['booking']['yy'],
			$data['booking']['mm'],
			$data['booking']['dd'],
			Helper::getMeridiem(
				$data['booking']['hh'],
				$data['booking']['ii']
			),
		]);
	}

	/**
	 * Display the specified resource.
	 *
	 * @param string $booking_code
	 */
	public function show(int $yy, int $mm, int $dd, string $booking_code)
	{
		// initialized
		$data = [];

		// get parameeters
		$data = [
			'yy' => $yy,
			'mm' => $mm,
			'dd' => $dd,
			'booking_code' => $booking_code,
		];
		//echo '<pre>';print_r($data);die();

		// get booking
		$data['booking'] = $this->booking->getBooking(
			$keys = [
				['booking_code', '=', $data['booking_code']],
			]
		);
		//echo '<pre>';print_r($data['booking']);die();

		// get geoip
		if($data['booking']['is_web']) {

			$data['geoip'] = GeoIP($ip = $data['booking']['agent_ip']);
			//$data['geoip'] = GeoIP($ip = '133.201.169.32');
			//echo '<pre>';print_r($data['geoip']);die();
		}

		// get count
		$data['count'] = [
			'accepted_numbers' => $this->master->getField(
				$field = 'accepted_numbers',
				$keys = [
					['yy', '=', $data['booking']['yy']],
					['mm', '=', $data['booking']['mm']],
					['dd', '=', $data['booking']['dd']],
				],
				$table = 'mtb_date'
			),
			'booked_numbers' => $this->booking->getCount(
				$keys = [
					['yy', '=', $data['booking']['yy']],
					['mm', '=', $data['booking']['mm']],
					['dd', '=', $data['booking']['dd']],
					['is_booked', '=', 1],
				],
				$table = 'dtb_booking'
			),
		];
		//echo '<pre>';print_r($data['count']);die();

		// set title
		$data['title'] = __("lang.booking.show");

		// get lang
		$data['lang'] = $this->lang;
		//echo '<pre>';print_r($data['lang']);die();

		// set tab
		$data['tab'] = 'booking';

		//echo '<pre>';print_r($data);die();
		return view('backend.covid.booking.show', $data);
	}

	/**
	 * Update the specified resource in storage.
	 *
	 * @param int $yy
	 * @param int $mm
	 * @param int $dd
	 * @param string $booking_code
	 */
	public function update(int $yy, int $mm, int $dd = 0, int $mtb_vaccine_id = 0, string $booking_code = null)
	{
		// initialized
		$data = [];

		// get parameeters
		$data = [
			'yy' => $yy,
			'mm' => $mm,
			'dd' => $dd,
			'mtb_vaccine_id' => $mtb_vaccine_id,
			'booking_code' => $booking_code,
		];
		//echo '<pre>';print_r($data);die();

		if($data['dd'] == 0) {

			// set keys
			$keys = ($data['mtb_vaccine_id'] == 0) ? [] : [
				['mtb_vaccine_id', '=', $data['mtb_vaccine_id']],
			];
			//echo '<pre>';print_r($keys);die();

			// append keys
			$keys = array_merge($keys, [
				['yy', '=', $data['yy']],
				['mm', '=', $data['mm']],
				['dd', '>=', 1],
			]);
			//echo '<pre>';print_r($keys);die();

			// get min_day
			$min_day = $this->service->getMin(
				$field = 'dd',
				$keys,
				$table = 'mtb_date'
			);
			//echo '<pre>';print_r($min_day);die();

			$data['dd'] = (is_null($min_day)) ? 1 : $min_day;
			//echo '<pre>';print_r($data['dd']);die();
		}

		// get date
		$data['date'] = $this->master->getDate(
			$keys = [
				['mtb_date.yy', '=', $data['yy']],
				['mtb_date.mm', '=', $data['mm']],
				['mtb_date.dd', '=', $data['dd']],
			]
		);
		//echo '<pre>';var_dump($data['date']);die();

		if(is_null($data['date'])) {

			$data['date'] = [
				'mtb_vaccine_id' => $data['mtb_vaccine_id'],
			];
		}
		//echo '<pre>';print_r($data['date']);die();

		// get date
		$data['date'] += [
			'last' => Helper::getDate(Carbon::create($data['yy'], $data['mm'], 1)->subMonth(1)->toDateString()),
			'next' => Helper::getDate(Carbon::create($data['yy'], $data['mm'], 1)->addMonth(1)->toDateString()),
			'day' => Carbon::create($data['yy'], $data['mm'], 1)->endOfMonth()->format('j'),
			'week' => Carbon::create($data['yy'], $data['mm'], 1)->startOfMonth()->format('w'),
		];
		//echo '<pre>';print_r($data['date']);die();

		// get count
		$data['count'] = [
			'accepted_numbers' => $this->master->getField(
				$field = 'accepted_numbers',
				$keys = [
					['yy', '=', $data['yy']],
					['mm', '=', $data['mm']],
					['dd', '=', $data['dd']],
				],
				$table = 'mtb_date'
			),
			'booked_numbers' => $this->booking->getCount(
				$keys = [
					['yy', '=', $data['yy']],
					['mm', '=', $data['mm']],
					['dd', '=', $data['dd']],
					['is_booked', '=', 1],
				],
				$table = 'dtb_booking'
			),
		];
		//echo '<pre>';print_r($data['count']);die();

		// get time_list
		$data['time_list'] = $this->master->getEnabledTimeList(
			$keys = [
				'yy' => $data['yy'],
				'mm' => $data['mm'],
				'dd' => $data['dd'],
				'is_booked' => 1,
			]
		);
		//echo '<pre>';print_r($data['time_list']);die();

		// set keys
		$keys = ($data['mtb_vaccine_id'] == 0) ? [] : [
			'mtb_vaccine_id' => $data['mtb_vaccine_id'],
		];
		//echo '<pre>';print_r($keys);die();

		// get booking_list
		$data['booking_list'] = $this->booking->getBookingCountList(
			$keys = array_merge($keys, [
				'yy' => $data['yy'],
				'mm' => $data['mm'],
			])
		);
		//echo '<pre>';print_r($data['booking_list']);die();

		// get vaccine_list
		$data['vaccine_list'] = $this->master->getMasterDropdownList(
			$keys = [],
			$orders = [],
			$table = 'vaccine'
		);
		//echo '<pre>';print_r($data['vaccine_list']);die();

		// get wareki_list
		$data['wareki_list'] = Helper::getWarekiYear();
		//echo '<pre>';print_r($data['wareki_list']);die();

		// get
		$data['numbers'] = Config::get('consts.numbers');
		//echo '<pre>';print_r($data['numbers']);die();

		if(!is_null($data['booking_code'])) {

			// get booking
			$data['booking'] = $this->booking->getBooking(
				$keys = [
					['booking_code', '=', $data['booking_code']],
				]
			);
			//echo '<pre>';print_r($data['booking']);die();

			// get time
			$data['time'] = $this->service->getFirst(
				$keys = [
					['hh', '=', $data['booking']['hh']],
					['ii', '=', $data['booking']['ii']],
					['mtb_date_id', '=', $data['date']['mtb_date_id']],
				],
				$table = 'mtb_time'
			);
			//echo '<pre>';print_r($data['time']);die();

			// set date_last
			$data['date_last'] = Helper::getDate($data['booking']['date_last']);

			// set birthday
			$data['birthday'] = Helper::getDate($data['booking']['birthday']);
		}

		// set form
		$data['form'] = [
			'url' => route('admin.covid.booking.store')
		];

		// set title
		$data['title'] = (is_null($booking_code)) ? __("lang.booking.create") : __("lang.booking.update");

		// get lang
		$data['lang'] = $this->lang;
		//echo '<pre>';print_r($data['lang']);die();

		// set tab
		$data['tab'] = 'booking';

		//echo '<pre>';print_r($data);die();
		return view('backend.covid.booking.update', $data);
	}

	/**
	 * Remove the specified resource from storage.
	 */
	public function destroy(int $yy, int $mm, int $dd, string $booking_code)
	{
		// initialized
		$data = [];

		// get parameters
		$data = [
			'yy' => $yy,
			'mm' => $mm,
			'dd' => $dd,
		];
		$this->booking->deleteRow(
			$keys = [
				['booking_code', '=', $booking_code]
			],
			$table = 'dtb_booking'
		);
		return redirect()->route('admin.covid.booking.index', $data);
	}
}
