<?php

namespace App\Http\Controllers\Backend\Covid\Export;

use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use App\Http\Requests\Backend\FrontdeskRequest;

// use phpoffice
use PhpOffice\PhpSpreadsheet\IOFactory,
	PhpOffice\PhpSpreadsheet\Spreadsheet,
	PhpOffice\PhpSpreadsheet\Worksheet\Drawing;

// use components
use	Carbon\Carbon,
	Config,
	Session,
	App\Helper;

// use services
use App\Services\ServiceInterface,
	App\Services\MasterInterface,
	App\Services\BookingInterface;

class DayController extends Controller
{
	protected
		$service,
		$master,
		$booking;

	protected
		$title;

	public function __construct(
		ServiceInterface $service,
		MasterInterface $master,
		BookingInterface $booking
	) {
		parent::__construct();

		$this->service = $service;
		$this->master = $master;
		$this->booking = $booking;

		$this->title = __("lang.search.title");
	}

	/**
	 * Display a listing of the resource.
	 */
	public function index(int $yy = null, int $mm = null, int $dd = null)
	{
		// initialized
		$data = [];

		// get parameters
		$data = [
			'yy' => (is_null($yy)) ? date('Y') : $yy,
			'mm' => (is_null($mm)) ? date('n') : $mm,
			'dd' => (is_null($dd)) ? date('j') : $dd,
		];
		//echo '<pre>';print_r($data);die();

		// get booking_list
		$data['booking_list'] = $this->booking->getBookingList(
			$keys = [
				['yy', '=', $data['yy']],
				['mm', '=', $data['mm']],
				['dd', '=', $data['dd']],
				['is_booked', '=', 1],
			],
			$orders = [
				'dd' => 'ASC',
				'hh' => 'ASC',
				'ii' => 'ASC',
			],
			$options = []
		);
		//echo '<pre>';print_r($data['booking_list']);die();

		$date = Carbon::create(
			$data['yy'],
			$data['mm'],
			$data['dd']
		);
		//echo '<pre>';print_r($date);die();

		$file_path = implode('/', [
			storage_path(),
			'app',
			'public',
			'excel',
			'booking.xlsx',
		]);
		//echo '<pre>';print_r($file_path);die();

		$reader = IOFactory::createReader('Xlsx');
		$spreadsheet = $reader->load($file_path);

		$sheet_name = $date->toDateString();
		$spreadsheet->getActiveSheet()->setTitle($sheet_name, false);

		// set current date
		$spreadsheet->getActiveSheet()
			->setCellValue("A1", $date->isoFormat('YYYY年MM月DD日（ddd）'));

		// get pref_list
		$pref_list = $this->master->getMasterDropdownList(
			$keys = [],
			$orders = [],
			$table = 'pref'
		);
		// dd($pref_list);

		$row = 4;
		foreach($data['booking_list'] as $key => $value) {
			// dd($value);

			$patient_skana = $value['patient_skana'];
			$patient_gkana = $value['patient_gkana'];

			if (preg_match('/^[ァ-ヾ]+$/u', $patient_skana)) $patient_skana = mb_convert_kana($patient_skana, 'c');
			if (preg_match('/^[ァ-ヾ]+$/u', $patient_gkana)) $patient_gkana = mb_convert_kana($patient_gkana, 'c');
			
			// set address
			$address = '';

			if ($value['mtb_pref_id'] != 0) {
				
				$address = implode('', [
					$pref_list[$value['mtb_pref_id']],
					$value['address1'],
					$value['address2'],
					$value['address3'],
				]);
			}

			$spreadsheet->getActiveSheet()
					->setCellValue("A$row", $key + 1)

					->setCellValue("C$row", Carbon::create(
						$value['yy'],
						$value['mm'],
						$value['dd'],
						$value['hh'],
						$value['ii'],
						0
					)->format('H:i'))

					->setCellValue("D$row", $address)

					->setCellValue("E$row", implode(' ', [
						$patient_skana,
						$patient_gkana,
					]))

					->setCellValue("F$row", Carbon::parse($value['birthday'])->format("Y年m月d日"))

					->setCellValue("G$row", Helper::getAge($value['birthday']))

					->setCellValue("H$row", (is_null($value['mobile'])) ? $value['tel'] : $value['mobile']);

			// incrementing count
			$row++;
		}

		// define header
		$file_name = '予約リスト';
		header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
		header('Content-Disposition: attachment;filename="' . $file_name . '.xlsx"');

		// download file
		$writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
		$writer->save('php://output');
	}

	/**
	 * Store a newly created resource in storage.
	 */
	public function store(Request $request)
	{
		//
	}

	/**
	 * Display the specified resource.
	 */
	public function show(int $id)
	{
		//
	}

	/**
	 * Update the specified resource in storage.
	 */
	public function update(Request $request, string $id)
	{
		//
	}

	/**
	 * Remove the specified resource from storage.
	 */
	public function destroy(string $id)
	{
		//
	}
}
