<?php

namespace App\Http\Controllers\Frontend\Covid\Booking\Mypage;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\Frontend\Covid\Booking\DatetimeRequest;

// use components
use Carbon\Carbon,
    Config,
    Session,
    App\Helper;

// use services
use App\Services\ServiceInterface,
    App\Services\MasterInterface,
    App\Services\BookingInterface;

class EditController extends Controller
{
    protected
        $service,
        $master,
        $booking;

    public function __construct(
        ServiceInterface $service,
        MasterInterface $master,
        BookingInterface $booking
    ) {
        parent::__construct();

        $this->service = $service;
        $this->master = $master;
        $this->booking = $booking;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // initialized
        $data = [];

        // set title
        $data['title'] = __("lang.booking.index");

        // get lang
        $data['lang'] = $this->lang;
        //echo '<pre>';print_r($data['lang']);die();

        // get booking
        $data['booking'] = Session::get('booking');

        //echo '<pre>';print_r($data);die();
        return view('frontend.covid.booking.mypage.edit.index', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // initialized
        $data = [];

        // get request
        $data = $request->all();
        //echo '<pre>';print_r($data);die();

        // get accepted_number
        $accepted_number = $this->service->getField(
            $field = 'accepted_number',
            $keys = [
                ['mtb_date_id', '=', $this->service->getField(
                    $field = 'mtb_date_id',
                    $keys = [
                        ['yy', '=', $data['edited_booking']['yy']],
                        ['mm', '=', $data['edited_booking']['mm']],
                        ['dd', '=', $data['edited_booking']['dd']],
                    ],
                    $table = 'mtb_date'
                )],
                ['hh', '=', $data['edited_booking']['hh']],
                ['ii', '=', $data['edited_booking']['ii']],
            ],
            $table = 'mtb_time'
        );
        //echo '<pre>';print_r($accepted_number);die();

        // get booking_count
        $booking_count = $this->service->getCount(
            $keys = [
                ['dtb_booking.yy', '=', $data['edited_booking']['yy']],
                ['dtb_booking.mm', '=', $data['edited_booking']['mm']],
                ['dtb_booking.dd', '=', $data['edited_booking']['dd']],
                ['dtb_booking.hh', '=', $data['edited_booking']['hh']],
                ['dtb_booking.ii', '=', $data['edited_booking']['ii']],
                //['dtb_booking.is_booked', '=', 1],
                ['dtb_booking.is_pencil', '=', 1],
            ],
            $table = 'dtb_booking'
        );
        //echo '<pre>';print_r($booking_count);die();

        // booking on
        if ($accepted_number > $booking_count) {
            //die('booking on');

            $birthday = Carbon::create(Session::get('booking.birthday'));
            $date_last = Carbon::create(Session::get('booking.date_last'));

            // set session for storeBooking
            $this->booking->setSession([
                'booking' => $data['edited_booking'],
                'birthday' => [
                    'yy' => $birthday->year,
                    'mm' => $birthday->month,
                    'dd' => $birthday->day,
                ],
                'date_last' => [
                    'yy' => $date_last->year,
                    'mm' => $date_last->month,
                    'dd' => $date_last->day,
                ],
            ]);

            // update
            /**
             * --------------------------------------------------
             * transaction
             * --------------------------------------------------
             */
            \DB::transaction(function () use ($data) {
                //echo '<pre>';print_r($data);die();

                $data['booking'] = $this->booking->storeBooking($data);
                //echo '<pre>';print_r($data['booking']);die();

                // // send sms
                $sms = $this->booking->sendSms(
                    $data = $data['booking'],
                    $is_changed = true,
                );

                //die('ok');
            });

            // redirect
            return redirect()->route('frontend.covid.booking.mypage.edit.complete');
        }
        // booking off
        else {
            //die('booking off');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show()
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update($yy = null, $mm = null, $dd = null, $hh = null, $ii = null)
    {
        // initialized
        $data = [];

        // get parameters
        $data = [
            'yy' => (is_null($yy)) ? date('Y') : $yy,
            'mm' => (is_null($mm)) ? date('n') : $mm,
            'dd' => $dd,
            'hh' => $hh,
            'ii' => $ii,
            'is_pm' => (is_null($hh)) ? 0 : Helper::getMeridiem($hh, $ii),
        ];
        //echo '<pre>';print_r($data);die();

        // get mtb_vaccine_id
		$mtb_vaccine_id = Session::get('booking.mtb_vaccine_id');

        // get enabled_date
		$data['enabled_date'] = Carbon::today()->addDay(2);
		//echo '<pre>';print_r($data['enabled_date']->toDateString());die();

        // get min_dd
		$min_dd = ($data['enabled_date']->copy()->format('n') == $data['mm']) ? $data['enabled_date']->copy()->format('j') : 1;
		//echo '<pre>';print_r($min_dd);die();

        // get dd
        if(is_null($data['dd'])) {

            // set keys
			$keys = (is_null($mtb_vaccine_id)) ? [] : [
				'mtb_vaccine_id' => $mtb_vaccine_id,
				//['mtb_vaccine_id', '=', $mtb_vaccine_id],
			];
			//echo '<pre>';print_r($keys);die();

            // append keys
            $keys = array_merge($keys, [
                ['yy', '=', $data['yy']],
                ['mm', '=', $data['mm']],
                ['dd', '>=', $min_dd],
            ]);
            //echo '<pre>';print_r($keys);die();

            // get min_day
            $min_day = $this->service->getMin(
                $field = 'dd',
                $keys,
                $table = 'mtb_date'
            );
            //echo '<pre>';print_r($min_day);die();

            $data['dd'] = (is_null($min_day)) ? 1 : $min_day;
        }
        //echo '<pre>';print_r($data);die();

        // get date
        $data['date'] = [
            'prev' => Helper::getDate(Carbon::create($data['yy'], $data['mm'], 1)->subMonth(1)->toDateString()),
            'next' => Helper::getDate(Carbon::create($data['yy'], $data['mm'], 1)->addMonth(1)->toDateString()),
            'day' => Carbon::create($data['yy'], $data['mm'], 1)->endOfMonth()->format('j'),
            'week' => Carbon::create($data['yy'], $data['mm'], 1)->startOfMonth()->format('w'),
        ];
        //echo '<pre>';print_r($data['date']);die();

        // get enabled_month
        $enabled_month = ($data['enabled_date']->copy()->format('n') <= $data['mm']) ? $data['mm'] : null;

        // get booking_list
        $data['booking_list'] = $this->booking->getBookingCountList(
            $keys = [
                'yy' => $data['yy'],
                'mm' => $enabled_month,
                'dd' => $min_dd,
                'mtb_vaccine_id' => $mtb_vaccine_id,
            ]
        );
        //echo '<pre>';print_r($data['booking_list']);die();

        // get time_list
        $data['time_list'] = Helper::getEnabledTimeList(
            $this->master->getEnabledTimeList(
                $keys = [
                    'yy' => $data['yy'],
                    'mm' => $enabled_month,
                    'dd' => $data['dd'],
                ]
            )
        );
        //echo '<pre>';print_r($data['time_list']);die();

        // set vaccine
        $data['vaccine'] = Helper::getVaccine(
            $this->master->getdate(
                $keys = [
                    ['yy', '=', $data['yy']],
                    ['mm', '=', $enabled_month],
                    ['dd', '=', $data['dd']],
                ],
            )
        );
        //echo '<pre>';print_r($data['vaccine']);die();

        // set is_past
        $data['is_past'] = Carbon::create($data['yy'], $data['mm'], 1)->lte(Carbon::today()->startOfMonth());

        // get booking
        $data['booking'] = Session::get('booking');
        //echo '<pre>';print_r($data);die();

        // set form
        $data['form'] = [
            'url' => route('frontend.covid.booking.mypage.edit.confirm'),
        ];

        // set title
        $data['title'] = __("lang.booking.index");

        // get lang
        $data['lang'] = $this->lang;
        //echo '<pre>';print_r($data['lang']);die();

        return view('frontend.covid.booking.mypage.edit.update', $data);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy()
    {
        /**
         * --------------------------------------------------
         * transaction
         * --------------------------------------------------
         */
        \DB::transaction(function () {
            //echo '<pre>';print_r($data);die();

            $booking_code = Session::get('booking.booking_code');

            $this->booking->deleteRow(
                $keys = [
                    ['booking_code', '=', $booking_code],
                ],
                $table = 'dtb_booking'
            );

            // // send sms
            $canceled_booking = Session::get('booking');

            $sms = $this->booking->sendSmsCancel($canceled_booking);

            //die('ok');
        });

        // redirect
        return redirect()->route('frontend.covid.booking.home.index');
    }

    /**
     * Store a newly created resource in storage.
     */
    // public function confirm(Request $request)
    public function confirm(DatetimeRequest $request)
    {
        // initialized
        $data = [];

        // get booking
        $data['booking'] = Session::get('booking');
        //echo '<pre>';print_r($data);die();

        // get edited_booking
        $data['edited_booking'] = $request->booking;

        // set vaccine
        $data['vaccine'] = Helper::getVaccine(
            $this->master->getdate(
                $keys = [
                    ['yy', '=', $data['edited_booking']['yy']],
                    ['mm', '=', $data['edited_booking']['mm']],
                    ['dd', '=', $data['edited_booking']['dd']],
                ],
            )
        );
        //echo '<pre>';print_r($data['vaccine']);die();

        $data['edited_booking']['mtb_vaccine_id'] = $data['vaccine']['mtb_vaccine_id'];

        // get available_date
        $available_date = Carbon::create(
            $data['booking']['date_last'],
        )->addMonths(3);
        //echo '<pre>';print_r($available_date);die();

        // get edited_booking_date
        $edited_booking_date = Carbon::create(
            $data['edited_booking']['yy'],
            $data['edited_booking']['mm'],
            $data['edited_booking']['dd'],
        );
        //echo '<pre>';print_r($booking_date);die();

        // get birthday
        $birthday = Carbon::create(
            $data['booking']['birthday']
        );


        // set form
        $data['form'] = [
            'url' => route('frontend.covid.booking.mypage.edit.store'),
        ];

        // set title
        $data['title'] = __("lang.booking.index");

        // get lang
        $data['lang'] = $this->lang;
        //echo '<pre>';print_r($data['lang']);die();

        // set min_age
        $min_age = 12;

        if ($birthday->diffInYears($edited_booking_date) < $min_age) {
            return redirect()->route('frontend.covid.booking.mypage.edit.error')
                ->withErrors("予約日時点で{$min_age}歳未満です。<br>予約日を再検討してください。");
        };

        if ($available_date->gt($edited_booking_date)) {
            return redirect()->route('frontend.covid.booking.mypage.edit.error')
                ->withErrors('前回の接種より3か月以内の予約日です。<br>予約日を再検討してください。');
        }

        // check eligibility special case 2023 spring
        [$has_eligibility, $err_messages] = Helper::hasEligibilitySpecialCase2023Spring(
            $edited_booking_date,
            $birthday,
            Session::get('booking.is_staff'),
            Session::get('booking.is_disease')
        );
        if (!$has_eligibility) {
            return redirect()->route('frontend.covid.booking.datetime.error')
                ->withErrors($err_messages);
        };

        //echo '<pre>';print_r($data);die();
        return view('frontend.covid.booking.mypage.edit.confirm', $data);
    }

    /**
     * Display the specified resource.
     */
    public function complete()
    {
        // initialized
        $data = [];

        // signin processing

        // set title
        $data['title'] = __("lang.booking.index");

        // get lang
        $data['lang'] = $this->lang;
        //echo '<pre>';print_r($data['lang']);die();

        // get booking
        $data['booking'] = Session::get('booking');

        // set vaccine
        $data['vaccine'] = Helper::getVaccine(
            $this->master->getdate(
                $keys = [
                    ['yy', '=', $data['booking']['yy']],
                    ['mm', '=', $data['booking']['mm']],
                    ['dd', '=', $data['booking']['dd']],
                ],
            )
        );
        //echo '<pre>';print_r($data['vaccine']);die();

        //echo '<pre>';print_r($data);die();
        return view('frontend.covid.booking.mypage.edit.complete', $data);
    }

    /**
     * Display the error message.
     */
    public function error()
    {
        // set title
        $data['title'] = __("lang.booking.index");

        // get lang
        $data['lang'] = $this->lang;
        //echo '<pre>';print_r($data['lang']);die();

        //echo '<pre>';print_r($data);die();
        return view('frontend.covid.booking.mypage.edit.error', $data);
    }
}
